<?php
/**
 * Core: The Builder class
 *
 * @link        https://oxyprops.com
 *
 * @package     OxyProps
 * @subpackage  Core
 * @author      Cédric Bontems <cedric@thewebforge.dev>
 * @since       1.5.0
 * @copyright   Copyright (c) 2022, Cédric Bontems
 * @license     https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * OxyProps\Inc\Builder.
 *
 * All you need about the site builder.
 *
 * @since 1.5.0 Creation.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 */
class Builder {

	use Singleton;

	/**
	 * The name of the detected active site builder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $name The name of the detected active site builder.
	 */
	private string $name;

	/**
	 * The adapted name of the plugin
	 *
	 * @var string
	 *
	 * @since  1.5.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */

	/**
	 * The name of the plugin adapted to the site builder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $adapted_name The adapted name of the plugin.
	 */
	private string $adapted_name;

	/**
	 * A short version of the adapted name of the plugin
	 *
	 * @var string
	 *
	 * @since  1.5.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */

	/**
	 * A short version of the adapted name of the plugin.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $short_adapted_name The short version of the adapted name of the plugin.
	 */
	private string $short_adapted_name;

	/**
	 * A boolean indicating if the active site builder is supported
	 *
	 * @var bool
	 *
	 * @since  1.5.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */

	/**
	 * A boolean indicating if the active site builder is supported.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var bool $supported True if the active site builder is supported.
	 */
	private bool $supported;

	/**
	 * Initialize Builder class.
	 *
	 * Called by the constructor. It retreives and populates all the class properties.
	 *
	 * @see https://developer.wordpress.org/reference/functions/wp_get_theme/
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	private function register() {
		$this->set_supported( true );
		$this->set_adapted_name( 'OxyProps' );
		$this->set_short_adapted_name( 'OxyProps' );

		$theme = wp_get_theme();

		switch ( true ) {
			case defined( 'CT_VERSION' ):
				$this->set_name( 'Oxygen' );
				break;
			case 'Bricks' === $theme->name || 'Bricks' === $theme->parent_theme:
				$this->set_adapted_name( 'BricksProps' );
				$this->set_short_adapted_name( 'BricksProps' );
				$this->set_name( 'Bricks' );
				break;
			case defined( '__BREAKDANCE_VERSION' ):
				$this->set_adapted_name( 'BDProps' );
				$this->set_short_adapted_name( 'BDProps' );
				$this->set_name( 'Breakdance' );
				break;
			default:
				$this->set_supported( false );
				$this->set_name( 'Unsupported Site Buidler' );
		}
	}

	// * Getters and Setters.

	/**
	 * Get the name of the detected active site builder.
	 *
	 * @since 1.5.0 Describe changes.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string Description.
	 */
	public function get_name(): string {
		return $this->name;
	}

	/**
	 * Set the name of the detected active site builder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $name Provide the name of the detected active site builder.
	 * @return object The builder instance.
	 */
	public function set_name( string $name ): object {
		$this->name = $name;

		return $this;
	}

	/**
	 * Get the adapted name of the plugin.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string The adapted name of the plugin.
	 */
	public function get_adapted_name(): string {
		return $this->adapted_name;
	}

	/**
	 * Set the adapted name of the plugin.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $adapted_name Provide the adapted name of the plugin.
	 * @return object The builder instance.
	 */
	public function set_adapted_name( string $adapted_name ): object {
		$this->adapted_name = $adapted_name;

		return $this;
	}

	/**
	 * Get a short version of the adapted name of the plugin.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string A short version of the adapted name of the plugin.
	 */
	public function get_short_adapted_name(): string {
		return $this->short_adapted_name;
	}

	/**
	 * Set a short version of the adapted name of the plugin.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $short_adapted_name Provide a short version of the adapted name of the plugin.
	 * @return object The builder instance.
	 */
	public function set_short_adapted_name( string $short_adapted_name ): object {
		$this->short_adapted_name = $short_adapted_name;

		return $this;
	}

	/**
	 * Get a boolean indicating if the active site builder is supported
	 *
	 * @return  boolean
	 */

	/**
	 * Get a boolean indicating if the active site builder is supported.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return bool True if the active site builder is supported.
	 */
	public function get_supported():string {
		return $this->supported;
	}

	/**
	 * Set a boolean indicating if the active site builder is supported.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param bool $supported Provide true if the active site builder is supported.
	 * @return object The builder instance.
	 */
	public function set_supported( bool $supported ): object {
		$this->supported = $supported;

		return $this;
	}
}
